'use strict';

import { TagsField } from './tagsfield.js';
import { disallowed_edit_tag_prefixes } from './disallowed_edit_tag_prefixes.js';
import { autocopy } from './autocopy.js';
import { add_thumbnail_preview } from './thumbnail_preview.js';

class Manage {
	constructor() {
		this.helper = false;
		if (document.querySelector('meta[name="helper"]')) {
			this.helper = true;
		}
		this.drawer = this.create_drawer();
		this.panels = this.create_panels();
		this.form = this.create_form();
		this.generate_tag_suggestions = false;
		//TODO use modifiers for keys so that select all/none can be ctrl+a and ctrl+d
		this.keys = {
			exit_management: 'Escape',
			toggle_management: 'e',
			add_tags: 't',
			remove_tags: 'r',
			generate_set: 's',
			select_all: '=',
			select_none: '-',
			remove_files: 'Delete',
			select_add: 'Shift',
			//TODO this would be nice if it could be Alt
			// but it seems like some browsers prioritize Alt and won't even let dragstart fire while it's being held
			select_negate: 'Control',
		};
		this.select_add = false;
		this.select_negate = false;

		// listener for shortcut keys
		window.addEventListener('keydown', e => {
			if (this.keys.exit_management == e.key) {
				if (document.documentElement.classList.contains('active-panel')) {
					this.hide_panels();
					return;
				}
				this.exit_management();
				return;
			}
			if (this.keys.select_add == e.key && !this.select_add) {
				this.select_add = true;
			}
			if (this.keys.select_negate == e.key && !this.select_negate) {
				this.select_negate = true;
			}
			// ignore other management keys if in an input
			if (
				'INPUT' == document.activeElement.tagName
				|| 'TEXTAREA' == document.activeElement.tagName
			) {
				return;
			}
			if (this.keys.toggle_management == e.key) {
				this.toggle_management();
				return;
			}
			// ignore following shortcut keys if not in management mode
			if (!document.documentElement.classList.contains('managing-files')) {
				return;
			}
			if (this.keys.select_all == e.key) {
				this.select_all();
				return;
			}
			if (this.keys.select_none == e.key) {
				this.select_none();
				return;
			}
			// ignore following shortcut keys if no files are selected
			if (0 == document.documentElement.getAttribute('data-selection-total')) {
				return;
			}
			if (this.keys.add_tags == e.key) {
				this.add_tags();
			}
			else if (this.keys.remove_tags == e.key) {
				this.remove_tags();
			}
			else if (this.keys.generate_set == e.key) {
				this.generate_set();
			}
			else if (this.keys.remove_files == e.key) {
				this.remove_files();
			}
		});
		window.addEventListener('keyup', e => {
			if (this.keys.select_add == e.key) {
				this.select_add = false;
			}
			if (this.keys.select_negate == e.key) {
				this.select_negate = false;
			}
		});

		// selection total
		this.selection_total = this.drawer.querySelector('#selected-files-total');
		this.selection_total.dataset.count = 0;

		// highlight selected files when hovering over drawer
		this.drawer.addEventListener('mouseover', e => {
			document.documentElement.classList.add('highlight-selected');
		});
		this.drawer.addEventListener('mouseout', e => {
			document.documentElement.classList.remove('highlight-selected');
		});

		// select/deselect listeners to thumbnails
		let thumbnails = document.querySelectorAll('.thumbnail');
		this.iterate_thumbnails(thumbnails, (thumbnail) => {
			this.add_thumbnail_listener(thumbnail);
		});

		// add management link to topmenu
		let manage_link = document.createElement('a');
		manage_link.id = 'manage-files-toggle';
		manage_link.innerText = 'Manage';
		let topmenu = document.querySelector('#topmenu');
		manage_link.style.opacity = '0';
		manage_link.style.transition = 'opacity 1000ms';
		topmenu.appendChild(manage_link);
		setTimeout(() => {
			manage_link.style.opacity = '';
		}, 10);

		// management link listener
		manage_link.addEventListener('click', () => {
			this.toggle_management();
		});

		// create tags field
		this.tags_field = new TagsField(disallowed_edit_tag_prefixes);
		// disallow meta tag suggestions
		this.tags_field.meta_tag_suggestions = false;
		// add classes to tags field components
		this.tags_field.preview.classList.add('tags-preview');
		// wrap tags field preview
		this.tags_field.preview_wrapper = document.createElement('div');
		this.tags_field.preview_wrapper.classList.add('tags-preview-wrapper');
		this.tags_field.preview_wrapper.append(this.tags_field.preview);

		// add a wrapper for tag suggestions list
		this.tag_suggestions_list_wrapper = document.createElement('div');
		this.tag_suggestions_list_wrapper.classList.add('tag-suggestions-list-wrapper');

		// on tags field focus move tag suggestions element to wrapper (and move wrapper to beginning of form)
		this.tags_field.input.addEventListener('focus', e => {
			this.form.insertBefore(this.tag_suggestions_list_wrapper, this.form.children[0]);
			this.tag_suggestions_list_wrapper.appendChild(this.tags_field.tag_suggestions_list);
		});

		// store reference to management in tags field
		this.tags_field.manage = this;
		// save handler
		this.tags_field.save = function() {
			if (this.input.value) {
				// commit any tag still in input
				this.add_tags(this.to_list(this.input.value));
				this.clear_input();
			}
			// current tags in current form tags input
			this.input.parentNode.querySelector('[name="tags"]').value = this.to_string(this.tags_list);
		}.bind(this.tags_field);
		// only fetch tag suggestions on first search input focus
		this.tags_field.input.addEventListener('focus', e => {
			if (e.currentTarget.hasOwnProperty('fetched')) {
				return;
			}
			e.currentTarget.fetched = true;
			// fetch suggestions
			this.tags_field.fetch_suggestions();
		});

		// management action buttons
		let actions = [
			'rebuild-files',
			'remove-files',
			'generate-set',
			'copy-tags',
			'add-tags',
			'remove-tags',
			'select-all',
			'select-none',
		];
		if (this.helper) {
			delete actions['rebuild-files'];
			delete actions['remove-files'];
			delete actions['remove-tags'];
		}
		for (let i = 0; i < actions.length; i++) {
			let action = actions[i];
			let action_button = this.drawer.querySelector('#manage-files-' + action);
			if (!action_button) {
				continue;
			}
			action_button.addEventListener('click', () => {
				this[action.replace('-', '_')]();
			});
		}
		// management panel buttons
		let panels = [
			//TODO no owner management for now
			//'owner',
			'publish-time',
		];
		if (this.helper) {
			delete panels['owner'];
			delete panels['publish-time'];
		}
		for (let i = 0; i < panels.length; i++) {
			let panel = panels[i];
			let panel_button = this.drawer.querySelector('#manage-files-' + panel);
			if (!panel_button) {
				continue;
			}
			panel_button.addEventListener('click', e => {
				this.toggle_panel(e.currentTarget.id.substring(13));
			});
		}

		// dim background for active panel
		this.dim = document.querySelector('#dim');
		if (!this.dim) {
			this.dim = document.createElement('div');
			this.dim.id = 'dim';
			document.body.appendChild(this.dim);
		}
		this.dim.addEventListener('click', (e) => {
			this.hide_panels();
		});

		// drag to select
		this.drag_origin = {
			x: 0,
			y: 0,
		};
		this.blank = document.createElement('img');
		// transparent pixel to hide generated ghost while dragging
		this.blank.src = 'data:image/gif;base64,R0lGODlhAQABAIAAAAAAAP///yH5BAEAAAAALAAAAAABAAEAAAIBRAA7';
		this.blank.id = 'blank';
		this.selection_box = document.createElement('span');
		this.selection_box.id = 'selection-box';
		this.hide_selection_box();
		document.body.appendChild(this.blank);
		document.body.appendChild(this.selection_box);
		document.body.addEventListener('dragstart', (e) => {
			if (
				!document.documentElement.classList.contains('managing-files')
				|| document.documentElement.classList.contains('active-panel')
			) {
				return;
			}
			this.show_selection_box();
			e.dataTransfer.setDragImage(this.blank, 0, 0);
			this.drag_origin.x = e.pageX;
			this.drag_origin.y = e.pageY;
		});
		document.body.addEventListener('drag', (e) => {
			if (document.documentElement.classList.contains('active-panel')) {
				return;
			}
			this.update_selection_box(e.pageX, e.pageY);
		});
		document.body.addEventListener('dragend', (e) => {
			e.preventDefault();
			if (document.documentElement.classList.contains('active-panel')) {
				return;
			}
			if (document.documentElement.classList.contains('managing-files')) {
				this.update_selection_box(e.pageX, e.pageY);
				this.selection_from_drag();
			}
			this.hide_selection_box();
		});

		this.form.addEventListener('submit', (e) => {
			console.log('submitting form data:');
			let fd = new FormData(this.form);
			for (let pair of fd.entries()){
				console.log(pair[0], pair[1]);
			}
			e.preventDefault();
			let tags_input = this.form.querySelector('input[name="tags"]');
			if (tags_input) {
				this.tags_field.save();
				this.generate_tag_suggestions = true;
			}
			let selected = document.querySelectorAll('.selected');
			this.iterate_thumbnails(selected, (thumbnail) => {
				this.api_request(
					'POST',
					this.form.action,
					new FormData(this.form),
					thumbnail
				);
			});
			this.hide_panels();
		});

		// add listener for leaving page to check if any requests are still processing
		window.addEventListener('beforeunload', e => {
			if (document.querySelector('.processing')) {
				(e || window.event).returnValue = true;
				return true;
			}
		});
	}
	add_processing(thumbnail) {
		// make sure processing animation plays from beginning
		thumbnail.style.animation = 'none';
		thumbnail.offsetHeight; // getting box metrics should trigger reflow
		setTimeout(() => {
			thumbnail.classList.add('processing');
			thumbnail.style.animation = '';
		}, 10);
	}
	add_thumbnail_listener(thumbnail) {
		thumbnail.addEventListener('click', (e) => {
			if (!document.documentElement.classList.contains('managing-files')) {
				return;
			}
			e.preventDefault();
			e.stopPropagation();
			if (document.documentElement.classList.contains('active-panel')) {
				this.hide_panels();
				return;
			}
			this.toggle_select(e.currentTarget)
			this.update_selection_total();
		});
	}
	clear_result(thumbnail) {
		thumbnail.classList.remove('success');
		thumbnail.classList.remove('failure');
	}
	toggle_select(thumbnail) {
		this.clear_result(thumbnail);
		thumbnail.classList.toggle('selected');
	}
	select(thumbnail) {
		this.clear_result(thumbnail);
		thumbnail.classList.add('selected');
	}
	deselect(thumbnail) {
		this.clear_result(thumbnail);
		thumbnail.classList.remove('selected');
	}
	hide_selection_box() {
		let items = [
			'blank',
			'selection_box',
		];
		for (let i = 0; i < items.length; i++) {
			let item = items[i];
			this[item].style.display = 'none';
			this[item].style.left = '0';
			this[item].style.top = '0';
			this[item].style.width = '0';
			this[item].style.height = '0';
		}
	}
	show_selection_box() {
		this.blank.style.display = 'inline-block';
		this.blank.style.width = '2px';
		this.blank.style.height = '2px';
		this.selection_box.style.display = 'inline-block';
	}
	update_selection_box(page_x, page_y) {
		if (page_x < this.drag_origin.x) {
			this.selection_box.style.left = page_x + 'px';
			this.selection_box.style.width = this.drag_origin.x - page_x + 'px';
		}
		else {
			this.selection_box.style.left = this.drag_origin.x + 'px';
			this.selection_box.style.width = page_x - this.drag_origin.x + 'px';
		}
		if (page_y < this.drag_origin.y) {
			this.selection_box.style.top = page_y + 'px';
			this.selection_box.style.height = this.drag_origin.y - page_y + 'px';
		}
		else {
			this.selection_box.style.top = this.drag_origin.y + 'px';
			this.selection_box.style.height = page_y - this.drag_origin.y + 'px';
		}
	}
	selection_from_drag() {
		// default drag-select replaces
		if (
			!this.select_add
			&& !this.select_negate
		) {
			this.select_none();
		}
		let r1 = this.selection_box.getBoundingClientRect();
		let thumbnails = document.querySelectorAll('.thumbnail');
		this.iterate_thumbnails(thumbnails, (thumbnail) => {
			let r2 = thumbnail.getBoundingClientRect()
			if (
				!(
					r2.left > r1.right
					|| r2.right < r1.left
					|| r2.top > r1.bottom
					|| r2.bottom < r1.top
				)
			) {
				if (this.select_negate) {
					this.deselect(thumbnail);
				}
				else {
					this.select(thumbnail);
				}
			}
		})
		this.update_selection_total();
	}
	enter_management() {
		this.select_none();
		this.hide_panels();
		document.documentElement.classList.add('managing-files');
		let filtered = document.querySelectorAll('.thumbnail.filtered');
		if (filtered) {
			this.iterate_thumbnails(filtered, thumbnail => {
				thumbnail.classList.add('not-filtered');
				thumbnail.classList.remove('filtered');
			});
		}
	}
	exit_management() {
		document.documentElement.classList.remove('managing-files');
		this.select_none();
		this.hide_panels();
		let filtered = document.querySelectorAll('.thumbnail.not-filtered');
		if (filtered) {
			this.iterate_thumbnails(filtered, thumbnail => {
				thumbnail.classList.add('filtered');
				thumbnail.classList.remove('not-filtered');
			});
		}
	}
	toggle_management() {
		document.documentElement.classList.toggle('managing-files');
		if (document.documentElement.classList.contains('managing-files')) {
			this.enter_management();
		}
		else {
			this.exit_management();
		}
	}
	show_panel(panel) {
		this.hide_panels();
		let panel_el = this.panels.querySelector('#manage-files-panel-' + panel);
		if (!panel_el) {
			return;
		}
		this.form.insertBefore(panel_el, this.form.firstChild);
		this.form.action = this.form.dataset.api + panel_el.dataset.endpoint;
		document.documentElement.classList.add('active-panel');
		this.form.querySelector('input[type="text"]').focus();
	}
	hide_panels() {
		document.documentElement.classList.remove('active-panel');
		let panels = this.form.querySelectorAll('.manage-files-panel');
		for (let i = panels.length - 1; 0 <= i; i--) {
			this.panels.appendChild(panels[i]);
		}
	}
	toggle_panel(panel) {
		if (this.form.querySelector('#manage-files-panel-' + panel)) {
			this.hide_panels();
		}
		else {
			this.show_panel(panel);
		}
	}
	iterate_thumbnails(thumbnails, cb) {
		let reversed_array = [];
		for (let i = thumbnails.length - 1; 0 <= i; i--) {
			reversed_array.push(thumbnails[i]);
		}
		reversed_array = reversed_array.reverse();
		for (let i = 0; i < reversed_array.length; i++) {
			cb(reversed_array[i]);
		}
	}
	api_request(method, action, fd, thumbnails, cb) {
		let xhr = new XMLHttpRequest();
		if (NodeList !== thumbnails.constructor) {
			thumbnails = [thumbnails];
		}
		else {
			thumbnails = Array.prototype.slice.call(thumbnails);
		}
		xhr.file_ids = [];
		this.iterate_thumbnails(thumbnails, (thumbnail) => {
			xhr.file_ids.push(thumbnail.dataset.id);
		});
		fd.append('file_ids', xhr.file_ids);
		xhr.thumbnails = thumbnails;
		xhr.cb = cb;
		fd.append('response_type', 'json');
		xhr.responseType = 'json';
		xhr.onreadystatechange = () => {
			if (xhr.readyState == XMLHttpRequest.DONE) {
				this.iterate_thumbnails(xhr.thumbnails, (thumbnail) => {
					thumbnail.classList.remove('processing');
				});
				if (this.generate_tag_suggestions) {
					let processing = document.querySelectorAll('.thumbnail.processing');
					if (!processing.length) {
						// send generate tag suggestions request
						let tags_xhr = new XMLHttpRequest();
						tags_xhr.open(
							'POST',
							action + (-1 != action.indexOf('?') ? '&' : '?') + '_' + new Date().getTime(),
							true
						);
						tags_xhr.withCredentials = true;
						//TODO generate tag suggestions
						let fd2 = new FormData();
						fd2.append('mode', 'generate_tag_suggestions');
						tags_xhr.send(fd2);
						this.generate_tag_suggestions = false;
					}
				}
				if (xhr.cb) {
					xhr.cb(xhr);
				}
				if (200 == xhr.status) {
					if (!xhr.response) {
						return;
					}
					if (xhr.response.hasOwnProperty('file_records')) {
						for (let file_id in xhr.response.file_records) {
							let thumbnail = document.querySelector('[data-id="' + file_id + '"]');
							if (!thumbnail) {
								continue;
							}
							let response_file = xhr.response.file_records[file_id];
							if (response_file.hasOwnProperty('failure')) {
								thumbnail.classList.add('failure');
							}
							else if (response_file.hasOwnProperty('remove')) {
								thumbnail.parentNode.removeChild(thumbnail);
							}
							else if (response_file.hasOwnProperty('thumbnail')) {
								let temp = document.createElement('div');
								temp.innerHTML = response_file.thumbnail;
								let new_thumbnail = temp.querySelector('.thumbnail');
								if (thumbnail.classList.contains('selected')) {
									new_thumbnail.classList.add('selected');
								}
								// add management listener
								this.add_thumbnail_listener(new_thumbnail);
								// preserve view url
								new_thumbnail.querySelector('a').href = thumbnail.querySelector('a').href;
								// maybe monkeypatch populate_file_properties from thalassa proper later?
								let new_summary = new_thumbnail.querySelector('.inner');
								// preserve filtered
								if (thumbnail.classList.contains('not-filtered')) {
									new_thumbnail.classList.add('not-filtered');
								}
								// replace entire thumbnail
								thumbnail.parentNode.insertBefore(new_thumbnail, thumbnail);
								thumbnail.parentNode.removeChild(thumbnail);
								// if thumbnail has clip then add hover preview listener
								if (new_thumbnail.dataset.hasOwnProperty('clip')) {
									add_thumbnail_preview(new_thumbnail);
								}
								xhr.thumbnails.push(new_thumbnail);
							}
						}
					}
					// only iterating original file ids, since additional thumbnails may have been returned with the response (like other files belonging to the same set)
					for (let i = 0; i < xhr.file_ids.length; i++) {
						let thumbnail = document.querySelector('.thumbnail[data-id="' + xhr.file_ids[i] + '"]');
						thumbnail.classList.add('success');
						setTimeout(() => {
							thumbnail.classList.remove('success');
						}, 2000);
					}
				}
				else {
					this.iterate_thumbnails(xhr.thumbnails, (thumbnail) => {
						thumbnail.classList.add('failure');
					});
				}
			}
		};
		xhr.original_file_ids = [];
		this.iterate_thumbnails(xhr.thumbnails, (thumbnail) => {
			this.clear_result(thumbnail);
			this.add_processing(thumbnail);
		});
		xhr.open(
			method,
			action + (-1 != action.indexOf('?') ? '&' : '?') + '_' + new Date().getTime(),
			true
		);
		xhr.withCredentials = true;
		xhr.send(fd);
	}
	rebuild_files() {
		if (this.helper) {
			return;
		}
		this.hide_panels();
		let selected = document.querySelectorAll('.selected');
		this.iterate_thumbnails(selected, (thumbnail) => {
			let fd = new FormData();
			this.api_request(
				'POST',
				this.form.dataset.api + 'rebuild_files',
				fd,
				thumbnail
			);
		});
	}
	remove_files() {
		if (this.helper) {
			return;
		}
		if (!confirm('This will permanently remove selected files, are you sure?')) {
			return;
		}
		this.hide_panels();
		let selected = document.querySelectorAll('.selected');
		this.iterate_thumbnails(selected, (thumbnail) => {
			let fd = new FormData();
			this.api_request(
				'POST',
				this.form.dataset.api + 'remove_files',
				fd,
				thumbnail
			);
		});
	}
	generate_set() {
		let fd = new FormData();
		let sync = false;
		let selected = document.querySelectorAll('.selected');
		for (let i = 0; i < selected.length; i++) {
			if (selected[i].querySelector('.set-indicators')) {
				sync = true;
				break;
			}
		}
		let generate_set_confirmation = 'This will generate a random set tag and apply it to all selected files, are you sure?';
		let sync_sets_confirmation = 'Some of the files you selected are already part of one or more sets. Would you like to sync sets on all selected files?';
		if (!sync) {
			if (!confirm(generate_set_confirmation)) {
				return;
			}
		}
		else if (confirm(sync_sets_confirmation)) {
			fd.append('sync', '1');
		}
		else if (!confirm(generate_set_confirmation)) {
			return;
		}
		this.hide_panels();
		this.api_request(
			'POST',
			this.form.dataset.api + 'generate_set',
			fd,
			selected
		);
	}
	copy_tags() {
		this.hide_panels();
		let selected = document.querySelectorAll('.selected');
		let tags = [];
		this.iterate_thumbnails(selected, (thumbnail) => {
			if (!thumbnail.dataset.tags) {
				return;
			}
			let current_tags = thumbnail.dataset.tags.split('#');
			for (let i = 0; i < current_tags.length; i++) {
				let current_tag = current_tags[i].trim();
				if (!current_tag || -1 != tags.indexOf(current_tag)) {
					continue;
				}
				tags.push(current_tag);
			}
		});
		if (0 == tags.length) {
			alert('No tags to copy');
			return;
		}
		tags.sort();
		let tags_string = '#' + tags.join(' #');
		autocopy(
			tags_string,
			'Tags copied to clipboard',
			'Copy raw tags manually'
		);
	}
	add_tags() {
		this.move_tags_field('add');
		this.toggle_panel('add-tags');
	}
	remove_tags() {
		if (this.helper) {
			return;
		}
		this.move_tags_field('remove');
		this.toggle_panel('remove-tags');
	}
	move_tags_field(mode) {
		let tags_panel = document.querySelector('#manage-files-panel-' + mode + '-tags');
		tags_panel.insertBefore(this.tags_field.input, tags_panel.firstChild);
		tags_panel.insertBefore(this.tags_field.preview_wrapper, tags_panel.firstChild);
		setTimeout(() => {
			this.tags_field.clear();
			this.tags_field.input.focus();
		}, 5);
	}
	select_all() {
		let thumbnails = document.querySelectorAll('.thumbnail');
		this.iterate_thumbnails(thumbnails, (thumbnail) => {
			this.select(thumbnail);
		});
		this.update_selection_total();
	}
	select_none() {
		let selected = document.querySelectorAll('.selected');
		this.iterate_thumbnails(selected, (thumbnail) => {
			this.deselect(thumbnail);
		});
		this.update_selection_total();
	}
	update_selection_total() {
		let selected = document.querySelectorAll('.selected');
		let thumbnails = document.querySelectorAll('.thumbnail');
		if (selected.length == thumbnails.length) {
			document.documentElement.dataset.selection_total = 'all';
			this.selection_total.innerText = this.selection_total.dataset.all;
		}
		else {
			document.documentElement.dataset.selection_total = selected.length;
			this.selection_total.innerText = document.documentElement.dataset.selection_total;
		}
	}
	create_drawer() {
		let drawer = document.createElement('div');
		drawer.id = 'manage-files-drawer';
		let actions_el = document.createElement('span');
		actions_el.id = 'manage-files-actions';
		drawer.appendChild(actions_el);
		let actions = {
//TODO no owner management for now
//			'owner': ['Owner', 'Change owner'],
			'publish-time': ['Publish time', 'Change publish time'],
			'rebuild-files': ['Rebuild', 'Rebuild files'],
			'remove-files': ['Remove', 'Remove files'],
			'generate-set': ['Set', 'Add selected files to a set (or synchronize sets among selected files)'],
			'copy-tags': ['Copy tags', 'Copy tags'],
			'add-tags': ['Add tags', 'Add tags'],
			'remove-tags': ['Remove tags', 'Remove tags'],
			'select-all': ['Select all', 'Select all'],
			'select-none': ['Select none', 'Select none'],
		};
		if (this.helper) {
			delete actions['owner'];
			delete actions['publish-time'];
			delete actions['rebuild-files'];
			delete actions['remove-files'];
			delete actions['remove-tags'];
		}
		for (let action in actions) {
			let action_el = document.createElement('a');
			action_el.id = 'manage-files-' + action;
			action_el.innerText = actions[action][0];
			action_el.title = actions[action][1];
			drawer.appendChild(action_el);
		}
		let selected_total_el = document.createElement('span');
		selected_total_el.id = 'selected-files-total';
		selected_total_el.dataset.all = 'All';
		selected_total_el.dataset.count = 0;
		selected_total_el.innerText = '0';
		drawer.appendChild(selected_total_el);

		document.body.appendChild(drawer);
		return drawer;
	}
	create_panels() {
		let panels_el = document.createElement('div');
		panels_el.id = 'manage-files-panels';

		let panel_ids = [
			'owner',
			'publish-time',
			'add-tags',
			'remove-tags',
		];
		if (this.helper) {
			delete panel_ids['owner'];
			delete panel_ids['publish-time'];
			delete panel_ids['remove-tags'];
		}

		let panels = {};
		for (let i in panel_ids) {
			let panel_id = panel_ids[i];
			let el = document.createElement('div');
			el.classList.add('manage-files-panel');
			el.id = 'manage-files-panel-' + panel_id;
			panels[panel_id] = el
		}

		let el = null;

		// owner
		panels['owner'].dataset.endpoint = 'change_owner';
		// label
		el = document.createElement('label');
		el.for = 'owner-id';
		// select
		let select_el = document.createElement('select');
		select_el.id = 'owner-id';
		select_el.name = 'owner_id';
		//TODO options with values of tokens and innerText of nicknames
		//TODO get token nicknames
		// options
		//for (let i in token_nicknames) {
			//let item = token_nicknames[i];
			//let token = item[0];
			//let nickname = item[1];
			//let option_el = document.createElement('option');
			//option_el.value = token;
			//option_el.innerText = nickname;
			//select_el.appendChild(option_el);
		//}
		panels['owner'].appendChild(select_el);
		// br
		panels['owner'].appendChild(document.createElement('br'));
		// submit
		el = document.createElement('input');
		el.type = 'submit';
		el.value = 'Change owner';
		panels['owner'].appendChild(el);
		//panels_el.appendChild(panels['owner']);

		// publish time
		panels['publish-time'].dataset.endpoint = 'set_publish_time';
		// label
		el = document.createElement('label');
		el.for = 'datetime';
		el.innerText = 'Publish datetime';
		panels['publish-time'].appendChild(el);
		// time input
		el = document.createElement('input');
		el.type = 'text';
		el.id = 'datetime';
		el.name = 'datetime';
		el.placeholder = 'Leave blank for current datetime';
		panels['publish-time'].appendChild(el);
		// br
		panels['publish-time'].appendChild(document.createElement('br'));
		// submit
		el = document.createElement('input');
		el.type = 'submit';
		el.value = 'Change publish time';
		panels['publish-time'].appendChild(el);
		panels_el.appendChild(panels['publish-time']);

		// add tags
		panels['add-tags'].dataset.endpoint = 'edit_files_tags';
		// tag mode
		el = document.createElement('input');
		el.type = 'hidden';
		el.name = 'tag_mode';
		el.value = 'add';
		panels['add-tags'].appendChild(el);
		// tags input
		el = document.createElement('input');
		el.type = 'hidden';
		el.name = 'tags';
		panels['add-tags'].appendChild(el);
		// submit
		el = document.createElement('input');
		el.type = 'submit';
		el.value = 'Add tags';
		panels['add-tags'].appendChild(el);
		panels_el.appendChild(panels['add-tags']);

		// remove tags
		panels['remove-tags'].dataset.endpoint = 'edit_files_tags';
		// tag mode
		el = document.createElement('input');
		el.type = 'hidden';
		el.name = 'tag_mode';
		el.value = 'remove';
		panels['remove-tags'].appendChild(el);
		// tags input
		el = document.createElement('input');
		el.type = 'hidden';
		el.name = 'tags';
		panels['remove-tags'].appendChild(el);
		// submit
		el = document.createElement('input');
		el.type = 'submit';
		el.value = 'Remove tags';
		panels['remove-tags'].appendChild(el);
		panels_el.appendChild(panels['remove-tags']);

		document.body.appendChild(panels_el);

		return panels_el;
	}
	create_form() {
		let form = document.createElement('form');
		form.id = 'manage-files-form';
		form.dataset.api = document.querySelector('link[rel=api]').href;
		form.action = '';
		form.method = 'POST';
		document.body.appendChild(form);
		return form;
	}
}

let manage = new Manage();
