'use strict';
import { TagsField } from './tagsfield.js';
import { disallowed_edit_tag_prefixes } from './disallowed_edit_tag_prefixes.js';
import { autocopy } from './autocopy.js';

// hide all non-summary children of .file and sets so that .file's width is only determined by the actual visual file
function hide_non_summary_file_elements() {
	let els = document.querySelectorAll('.file > *:not(.summary)');
	for (let i = 0; i < els.length; i++) {
		els[i].style.display = 'none';
	}
	let sets = document.querySelectorAll('.set');
	for (let i = 0; i < sets.length; i++) {
		sets[i].style.display = 'none';
	}
}
// show all non-summary children of .file and sets again
function show_non_summary_file_elements() {
	let els = document.querySelectorAll('.file > *:not(.summary)');
	for (let i = 0; i < els.length; i++) {
		els[i].style.display = '';
	}
	let sets = document.querySelectorAll('.set');
	for (let i = 0; i < sets.length; i++) {
		sets[i].style.display = '';
	}
}

// constrain some view page elements to the width of the file summary
function constrain_file_container_width() {
	let file_el = document.querySelector('.file');
	let file_el_width = file_el.clientWidth;
	file_el.style.maxWidth = 'calc(' + file_el_width + 'px)';
}

// fit visual file vertically
function fit_vertically(element) {
	let file = document.querySelector('.file');
	let available_height = window.innerHeight;
	// relying on #content having padding of 1rem which this script shouldn't really have to know about oops
	let rem = parseFloat(getComputedStyle(document.documentElement).fontSize);
	available_height -= (2.5 * rem);
	// subtracting topmenu height which this script also shouldn't have to know about
	let topmenu = document.querySelector('#topmenu');
	if (topmenu) {
		available_height -= topmenu.clientHeight;
	}
	// ignore all .file children that aren't .summary .file-info or .tags when calculating needed height
	let element_height = element.clientHeight;
	let summary_el = file.querySelector('.summary');
	let summary_height = 0;
	if (summary_el) {
		summary_height = summary_el.clientHeight;
	}
	let file_info_el = file.querySelector('.file-info');
	let file_info_height = 0;
	if (file_info_el) {
		file_info_height = file_info_el.clientHeight;
	}
	let tags_el = file.querySelector('.tags');
	let tags_height = 0;
	if (tags_el) {
		tags_height = tags_el.clientHeight;
	}
	let needed_height = summary_height + file_info_height + tags_height;

	if (needed_height > available_height) {
		let element_max_height = available_height;
		// subtract (.summary height - image height), .file_info, and .tags height
		element_max_height -= ((summary_height - element_height) + file_info_height + tags_height);
		element.style.height = element_max_height + 'px';
		element.style.width = 'auto';
	}
};

// run when file is done loading
function file_loaded(element) {
	constrain_file_container_width();
	show_non_summary_file_elements();
	if (element) {
		if (Number(localStorage.getItem('fit-files-vertically'))) {
			setTimeout(() => { fit_vertically(element); }, 100);
		}
	}
}

hide_non_summary_file_elements();

let waiting_target = document.querySelector('.file .summary img');
if (!waiting_target) {
	//TODO check for video element to use instead maybe?
	show_non_summary_file_elements();
}
else if (waiting_target.complete) {
	//TODO firefox delay?
	if (-1 != navigator.userAgent.toLowerCase().indexOf('firefox')) {
			file_loaded(waiting_target);
	}
	else {
		file_loaded(waiting_target);
	}
}
else {
	waiting_target.onload = () => {
		file_loaded(waiting_target);
	};
}

// thumbnail previews
import { add_thumbnail_preview } from './thumbnail_preview.js';
let thumbnails = document.querySelectorAll('.thumbnail');
for (let i = 0; i < thumbnails.length; i++) {
	let thumbnail = thumbnails[i];
	add_thumbnail_preview(thumbnail);
}

// management

// admin tags field
let real_tags_field = document.querySelector('#tags');
if (!real_tags_field) {
	// helper tags field
	real_tags_field = document.querySelector('#add-tags');
}
if (real_tags_field) {
	let tags_form = real_tags_field.parentNode;

	// create tags field
	let tags_field = new TagsField(disallowed_edit_tag_prefixes);
	// disallow meta tag suggestions
	tags_field.meta_tag_suggestions = false;
	// add classes to tags field components
	tags_field.preview.classList.add('tags-preview');
	// wrap tags field preview
	tags_field.preview_wrapper = document.createElement('div');
	tags_field.preview_wrapper.classList.add('tags-preview-wrapper');
	tags_field.preview_wrapper.append(tags_field.preview);

	// add a wrapper for tag suggestions list
	let tag_suggestions_list_wrapper = document.createElement('div');
	tag_suggestions_list_wrapper.classList.add('tag-suggestions-list-wrapper');

	// move tag suggestions element to wrapper (and move wrapper to beginning of form)
	tag_suggestions_list_wrapper.appendChild(tags_field.tag_suggestions_list);
	tags_form.insertBefore(tag_suggestions_list_wrapper, tags_form.children[0]);

	// add real tags field to tags field
	tags_field.real_tags_field = real_tags_field;

	// prevent regular tags form submit
	real_tags_field.parentNode.addEventListener('submit', e => {
		tags_field.save();
		return true;
	});

	// save handler
	tags_field.save = function() {
		if (this.input.value) {
			// commit any tag still in input
			this.add_tags(this.to_list(this.input.value));
			this.clear_input();
		}
		// current tags in current form tags input
		this.real_tags_field.value = this.to_string(this.tags_list);
	}.bind(tags_field);

	// only fetch tag suggestions on first search input focus
	tags_field.input.addEventListener('focus', e => {
		if (e.currentTarget.hasOwnProperty('fetched')) {
			return;
		}
		e.currentTarget.fetched = true;
		// fetch suggestions
		tags_field.fetch_suggestions();
	});

	// add tags field to tags form
	let tags_field_wrapper = document.createElement('div');
	tags_field_wrapper.classList.add('tags-field-wrapper');
	tags_field_wrapper.appendChild(tags_field.preview_wrapper);
	tags_field_wrapper.appendChild(tags_field.input);
	tags_form.appendChild(tags_field_wrapper);
	tags_form.appendChild(tags_form.querySelector('input[type="submit"]'));

	// populate existing tags
	let existing_tags = real_tags_field.value;
	existing_tags = tags_field.to_list(existing_tags);
	tags_field.add_tags(existing_tags);

	tags_field.input.classList.add('tag-field');

	// hide original tags field and label
	let label = tags_form.querySelector('label');
	real_tags_field.style.display = 'none';
	label.style.display = 'none';

	if (0 == tags_field.tags_list.length) {
		tags_field.preview_wrapper.classList.add('empty');
	}
	// add listener to swap negation and regular tags
	tags_field.input.addEventListener('added', e => {
		tags_field.preview_wrapper.classList.remove('empty');
	});
	tags_field.input.addEventListener('clear', e => {
		tags_field.preview_wrapper.classList.add('empty');
	});
	tags_field.input.addEventListener('removed', e => {
		if (0 == tags_field.tags_list.length) {
			tags_field.preview_wrapper.classList.add('empty');
		}
	});
	// copy tags
	let copy_tags_container = document.createElement('p');
	copy_tags_container.id = 'copy-tags';
	let copy_tags_link = document.createElement('a');
	copy_tags_link.innerText = 'Copy tags';
	copy_tags_link.addEventListener('click', e => {
		autocopy(tags_field.to_string(tags_field.tags_list), 'Tags copied to clipboard', 'Copy raw tags manually');
	});
	copy_tags_container.appendChild(copy_tags_link);
	tags_form.insertBefore(copy_tags_container, tags_form.firstChild);
}
