<?php 
if (!is_file('installed')) {
	require('thalassa_install.php');
	exit();
}

# php interface to thalassa_cli.py

$config = json_decode(file_get_contents('thalassa_site_config.json'), true);

$request_scheme = $config['force_request_scheme'];
if ('' == $request_scheme) {
	$request_scheme = $_SERVER['REQUEST_SCHEME'];
}
$config['request_scheme'] = $request_scheme;

$config['cookie_domain'] = $_SERVER['SERVER_NAME'];
$config['cookie_secure'] = false;
if ('https' == $config['request_scheme']) {
	$config['cookie_secure'] = true;
}
$config['cookie_lifespan'] = time() + 63072000;

define('CONFIG', $config);
define('THALASSA_FILEPATHS_SEPARATOR', '///');

# base thalassa command
$thalassa_cmd = CONFIG['python3_path'] . ' ' . CONFIG['thalassa_cli_path'] . ' --config=' . CONFIG['thalassa_config_path'];

# check for existing cookie token
$current_token = '';
$current_token_arg = '';
if (array_key_exists(CONFIG['cookie_name'], $_COOKIE)) {
	$current_token = $_COOKIE[CONFIG['cookie_name']];
}
# posted token overrides cookie token
if ('POST' == $_SERVER['REQUEST_METHOD'] && in_array('use_token', $_POST)) {
	$current_token = $_POST['use_token'];
}
# header token overrides cookie token
else if (in_array('HTTP_X_USE_TOKEN', $_SERVER)) {
	$current_token = $_SERVER['HTTP_X_USE_TOKEN'];
}
define('CURRENT_TOKEN', $current_token);
$current_token_arg = ' --use-token=' . escapeshellarg($current_token);

# get initial thalassa info (and check for any errors initializing thalassa)
$result_output = null;
$result_code = null;
exec($thalassa_cmd . $current_token_arg . ' --info', $result_output, $result_code);
if (0 != $result_code) {
	http_response_code(500);
	print('Problem initializing thalassa');
	exit();
}
$thalassa_info = [];
foreach ($result_output as $line) {
	$line_r = explode("\t", $line);
	if (1 < count($line_r)) {
		$key = $line_r[0];
		$value = $line_r[1];
		$thalassa_info[$key] = $value;
	}
}
define('THALASSA_INFO', $thalassa_info);
$result_output = null;
$result_code = null;

# authorized and final thalassa command
$authorized = false;
$token_mode = '';
if (array_key_exists('authorized', THALASSA_INFO) && '1' == THALASSA_INFO['authorized']) {
	$authorized = true;
	if (array_key_exists('token_mode', THALASSA_INFO)) {
		$token_mode = THALASSA_INFO['token_mode'];
	}
	$thalassa_cmd .= $current_token_arg;
}
define('AUTHORIZED', $authorized);
define('TOKEN_MODE', $token_mode);
define('THALASSA_CMD', $thalassa_cmd);
# pass set
$pass_set = false;
if (array_key_exists('pass_set', THALASSA_INFO) && '1' == THALASSA_INFO['pass_set']) {
	$pass_set = true;
}
define('PASS_SET', $pass_set);

# expose tags always for authorized, and only if configured otherwise
define('EXPOSE_TAGS', (AUTHORIZED || CONFIG['expose_tags']));

# root thalassa uri
$thalassa_uri = CONFIG['thalassa_uri'];
if ('' == $thalassa_uri) {
	$thalassa_uri = CONFIG['request_scheme'] . '://' . $_SERVER['SERVER_NAME'];
	if (!in_array($_SERVER['SERVER_PORT'], [80, 443])) {
		$thalassa_uri .= ':' . $_SERVER['SERVER_PORT'];
	}
	$thalassa_uri .= $_SERVER['PHP_SELF'];
	$thalassa_uri = str_replace('thalassa_web.php', '', $thalassa_uri);
	$thalassa_uri = str_replace('thalassa_api.php', '', $thalassa_uri);
}
define('THALASSA_URI', $thalassa_uri);

# static uri
$static_uri = THALASSA_URI . 'static/';
if ('' != CONFIG['static_uri']) {
	$static_uri = CONFIG['static_uri'];
}
define('STATIC_URI', $static_uri);

# files uri
$files_uri = THALASSA_URI . 'files/';
if ('' != CONFIG['files_uri']) {
	$files_uri = CONFIG['files_uri'];
}
define('FILES_URI', $files_uri);

# parse query array
$query_array = null;
parse_str($_SERVER['QUERY_STRING'], $query_array);
define('QUERY_ARRAY', $query_array);

# atom format
define('DATE_ATOM_W_TZ', 'Y-m-d\TH:i:s\ZP');

# max perpage for non-staff
define('MAX_PERPAGE_NONSTAFF', 128);

# general functions used for both web and api
function minify($res) {
	$res = str_replace("\n", '', $res);
	$res = str_replace("\t", '', $res);
	while (false !== strpos($res, '  ')) {
		$res = str_replace('  ', ' ', $res);
	}
	return $res;
}
function format_tags_string($tags_string) {
	$tags_string = trim($tags_string, "# \t\n");
	if (empty($tags_string)) {
		return '';
	}
	if (false === strpos($tags_string, '#')) {
		return '#' . $tags_string;
	}
	$tags_r = explode('#', $tags_string);
	$tags_string = '';
	foreach ($tags_r as $tag) {
		$tag = trim($tag, "\t\n");
		if ('' != $tag) {
			$tags_string .= ' #' . $tag;
		}
	}
	return trim($tags_string);
}
function send_response($res, $response_code=200) {
	http_response_code($response_code);
	print($res);
	exit();
}
function initialize_post_keys($keys) {
	foreach ($keys as $key) {
		if (!array_key_exists($key, $_POST)) {
			$_POST[$key] = '';
		}
	}
}
function preprocess_uploaded_files($files) {
	$temp_file_directory_path = THALASSA_INFO['config_files_directory_path'] . 'temp/';
	$temp_file_paths = [];
	$problem_upload_files = [];
	$filenames = [];
	# a frail attempt at avoiding temp name collisions when many files are uploaded at the same time
	$new_temp_file_string = time() . rand(0, 9) . rand(0, 9) . rand(0, 9);
	$file_count = count($files['name']);
	for ($i = 0; $i < $file_count; $i++) {
		$filename = $files['name'][$i];
		array_push($filenames, $filename);
		$temp_file_path = $files['tmp_name'][$i];
		$new_temp_file_name = $new_temp_file_string . $i;
		# supposed extension
		$supposed_extension = '.unknown';
		# last dot in filename
		$dot_index = strrpos($filename, '.');
		if (false !== $dot_index) {
			$supposed_extension = substr($filename, $dot_index);
		}
		$new_temp_file_path = $temp_file_directory_path . $new_temp_file_name . $supposed_extension;
		if (!is_uploaded_file($temp_file_path)) {
			array_push($problem_upload_files, [$filename, 'Not an uploaded file']);
		}
		else if (move_uploaded_file($temp_file_path, $new_temp_file_path)) {
			array_push($temp_file_paths, $new_temp_file_path);
		}
		else {
			array_push($problem_upload_files, [$filename, 'Unknown error']);
		}
	}
	return [$temp_file_paths, $problem_upload_files, $filenames];
}
function preprocess_fetched_files($file_urls) {
	$temp_file_directory_path = THALASSA_INFO['config_files_directory_path'] . 'temp/';
	$temp_file_paths = [];
	$problem_fetch_files = [];
	$filenames = [];
	# a frail attempt at avoiding temp name collisions when many files are fetched at the same time
	$new_temp_file_string = time() . rand(0, 9) . rand(0, 9) . rand(0, 9);
	foreach ($file_urls as $file_url) {
		if ('' == $file_url) {
			continue;
		}
		$supposed_extension = '.' . pathinfo($file_url, PATHINFO_EXTENSION);
		if (empty($supposed_extension)) {
			$supposed_extension = '.unknown';
		}
		$filename = pathinfo($file_url, PATHINFO_FILENAME) . $supposed_extension;
		array_push($filenames, $filename);
		$new_temp_file_path = $temp_file_directory_path . $new_temp_file_string . $supposed_extension;
		try {
			file_put_contents($new_temp_file_path, fopen($file_url, 'r'));
		}
		catch (Exception $e) {
			array_push($problem_fetch_files, $file_url);
		}
		array_push($temp_file_paths, $new_temp_file_path);
	}
	return [$temp_file_paths, $problem_fetch_files, $filenames];
}
function parse_file_record_string($file_record_string) {
	$file_record_r = explode("\t", $file_record_string);
	$file_record = [
		'file_id' => $file_record_r[0],
		'mimetype' => $file_record_r[1],
		'extension' => $file_record_r[2],
		'category' => $file_record_r[3],
		'size' => $file_record_r[4],
		'width' => $file_record_r[5],
		'height' => $file_record_r[6],
		'duration' => $file_record_r[7],
		'time' => $file_record_r[8],
		'owner' => $file_record_r[9],
		'tags' => [],
		'thumbnail' => $file_record_r[11],
		'thumbnail_clip' => $file_record_r[12],
		'cover_file_id' => $file_record_r[13],
		'sets' => [],
	];
	# tags
	$tags_r = array_filter(explode('#', trim(format_tags_string($file_record_r[10]), '# ')));
	$tags = [];
	foreach ($tags_r as $tag) {
		array_push($tags, trim($tag));
	}
	$file_record['tags'] = $tags;
	# sets
	$sets_r = array_filter(explode('#', trim(format_tags_string($file_record_r[14]), '# ')));
	$sets = [];
	while (count($sets_r)) {
		$set_id = trim(array_shift($sets_r));
		$set_file_ids = array_filter(explode(',', trim(array_shift($sets_r), ', ')));
		$sets[$set_id] = $set_file_ids;
	}
	$file_record['sets'] = $sets;
	return $file_record;
}
function parse_search_files_output($output) {
	$search_files_result = [];
	$search_files_result['total_results'] = array_shift($output);
	$search_files_result['total_pages'] = array_shift($output);
	$search_files_result['total_size'] = array_shift($output);
	$search_files_result['results_per_page'] = array_shift($output);
	$search_files_result['results_this_page'] = array_shift($output);
	$search_files_result['file_ids'] = [];
	$search_files_result['file_records'] = [];

	if (0 < count($output)) {
		$file_records = json_decode($output[0], true);
		foreach ($file_records as $file_record) {
			# file_ids should only have total_results items (other file_records are supplementary, e.g. cover images)
			if ($search_files_result['total_results'] > count($search_files_result['file_records'])) {
				array_push($search_files_result['file_ids'], $file_record['file_id']);
			}
			$search_files_result['file_records'][$file_record['file_id']] = $file_record;
		}
	}
	return $search_files_result;
}
function declutter_tags($tags) {
	$semantic_tags = [
		#'title:',
		'filename:',
		#'r18',
		'hidden',
		'focus:',
		'set:',
		'cover:',
		'mirror:',
		'superior of:',
		'inferior of:',
		'description:',
		'text:',
		'embed:',
		'_:',
		'audit:',
	];
	$decluttered_tags = [];
	foreach ($tags as $tag) {
		$include_tag = true;
		foreach ($semantic_tags as $semantic_tag) {
			if ($semantic_tag == substr($tag, 0, strlen($semantic_tag))) {
				$include_tag = false;
				break;
			}
		}
		if (!$include_tag) {
			continue;
		}
		if (!in_array($tag, CONFIG['clutter_tags'])) {
			array_push($decluttered_tags, $tag);
		}
	}
	return $decluttered_tags;
}

# render functions used for both web and api
function render_thumbnail($file_record, $file_records=[], $feed_thumbnail=false) {
	$thumbnail_file_record = null;
	# cover file specified, and it existed in file_records, and it had a renderable thumbnail
	if (
		'' != $file_record['cover_file_id']
		&& array_key_exists($file_record['cover_file_id'], $file_records)
		&& $file_records[$file_record['cover_file_id']]['thumbnail']
	) {
		# use cover file record for thumbnail
		$thumbnail_file_record = $file_records[$file_record['cover_file_id']];
	}
	# actual file record had a renderable thumbnail
	elseif ($file_record['thumbnail']) {
		$thumbnail_file_record = $file_record;
	}
	# thumbnail
	$res = '
		<div 
			class="thumbnail" 
			data-id="' . $file_record['file_id'] . '" 
			data-category="' . $file_record['category'] . '" 
			data-mimetype="' . $file_record['mimetype'] . '" 
			data-size="' . $file_record['size'] . '" 
			data-width="' . $file_record['width'] . '" 
			data-height="' . $file_record['height'] . '" 
			data-duration="' . $file_record['duration'] . '" 
			data-thumbnail="' . $file_record['thumbnail'] . '" 
			data-thumbnail-clip="' . $file_record['thumbnail_video_clip'] . '" 
			data-summary="' . $file_record['summary'] . '" ';
	# add data attribute to hidden files
	if (in_array('hidden', $file_record['tags'])) {
		$res .= '
			data-hidden="1" ';
	}
	# add data attribute to r18 files
	if (in_array('r18', $file_record['tags'])) {
		$res .= '
			data-r18="1" ';
	}
	# tags
	if ($file_record['tags'] && !$feed_thumbnail) {
		# always include tags in data attribute
		$tags = str_replace('<', '&lt;', $file_record['tags']);
		$res .= 'data-tags="' . str_replace('"', '&quot;', format_tags_string(implode('#', $tags))) . '" ';
		# only include tags in title if expose_tags is set
		if (EXPOSE_TAGS) {
			$res .= 'title="' . str_replace('"', '&quot;', format_tags_string(implode('#', declutter_tags($tags)))) . '" ';
		}
	}
	# is there a preview clip for this thumbnail
	if ($thumbnail_file_record) {
		#TODO maybe also make clips of very large gifs instead of always using them
		if ('image/gif' == $thumbnail_file_record['mimetype'] && 1 < $thumbnail_file_record['duration']) {
			$res .= 'data-clip="' . FILES_URI . 'original/' . $thumbnail_file_record['file_id'] . '.' . $thumbnail_file_record['extension'] . '"';
		}
		elseif ($thumbnail_file_record['thumbnail_video_clip']) {
			$res .= 'data-clip="' . FILES_URI . 'thumbnail/' . $thumbnail_file_record['file_id'] . '.webm"';
		}
	}
	$res .= '>';
	# sets
	if (count($file_record['sets']) && !$feed_thumbnail) {
		$res .= '<div class="set-indicators">';
		foreach ($file_record['sets'] as $set_id => $file_ids) {
			$res .= '<div class="set-indicator" title="' . $set_id . '">' . count($file_ids) . '</div>';
		}
		$res .= '</div>';
	}
	# inner
	if (!$thumbnail_file_record || !$thumbnail_file_record['thumbnail']) {
		$res .= '
	<span class="inner placeholder">
		<a href="' . THALASSA_URI . 'view/' . $file_record['file_id'] . '">';
		# category placeholder image
		$category_image = STATIC_URI . 'style/file_category_' . $file_record['category'];
		$res .= '
			<picture>
				<source 
					srcset="' . $category_image . '.svg" 
					type="image/svg"/>
				<img src="' . $category_image . '.svg" alt=""/>
			</picture>';
	}
	else {
		$res .= '
	<span class="inner">
		<a href="' . THALASSA_URI . 'view/' . $file_record['file_id'] . '">
			<picture>
				<source 
					srcset="' . FILES_URI . 'thumbnail/' . $thumbnail_file_record['file_id'] . '.' . $thumbnail_file_record['thumbnail'] . '" ';
		if ('webp' == $thumbnail_file_record['thumbnail']) {
			$res .= '
					type="image/webp"/>';
		}
		elseif ('jpg' == $thumbnail_file_record['thumbnail']) {
			$res .= '
					type="image/jpeg"/>';
		}
		$focus_style = '';
		# need to omit focus for feed thumbnail for some reason?
		if (!$feed_thumbnail) {
			# check if thumbnail has specified focus
			foreach ($thumbnail_file_record['tags'] as $tag) {
				if ('focus:' == substr($tag, 0, 6)) {
					$focus_value = intval(substr($tag, 6));
					$focus_style = 'style="object-position: ';
					if ($thumbnail_file_record['height'] > $thumbnail_file_record['width']) {
						$focus_style .= 'center ' . $focus_value . '%';
					}
					else {
						$focus_style .= $focus_value . '% center';
					}
					$focus_style .= ';"';
					break;
				}
			}
		}
		$res .= '
				<img src="' . FILES_URI . 'thumbnail/' . $thumbnail_file_record['file_id'] . '.' . $thumbnail_file_record['thumbnail'] . '" alt=""' . $focus_style . '/>
			</picture>';
	}
	$res .= '
				</a>
			</span>
		</div>';

	return $res;
}

function render_summary($file_record, $file_records=[], $feed_summary=false) {
	$summary_file_record = null;
	# cover file specified, and it existed in file_records, and it had a renderable summary
	if (
		'' != $file_record['cover_file_id']
		&& array_key_exists($file_record['cover_file_id'], $file_records)
#UNUSED		&& $file_records[$file_record['cover_file_id']]['summary']
	) {
		# use cover file record for summary
		$summary_file_record = $file_records[$file_record['cover_file_id']];
	}
	# actual file record had a renderable summary
	else {
#UNUSED if ($file_record['summary']) {
		$summary_file_record = $file_record;
	}

	return $res;
}

# functions to interface with thalassa cli
function check_pass($pass) {
	$pass_arg = '--current-pass=' . escapeshellarg($pass);
	$result_code = null;
	$result_output = null;
	exec(THALASSA_CMD . ' --check-pass ' . $pass_arg, $result_output, $result_code);
	return [$result_code, $result_output];
}
function change_pass($current_pass, $new_pass, $new_pass_confirmation) {
	$result_code = -1;
	if (!$current_pass) {
		$result_output = ['Current pass not entered'];
	}
	elseif (!$new_pass || !$new_pass_confirmation) {
		$result_output = ['New pass not entered'];
	}
	elseif ($new_pass != $new_pass_confirmation) {
		$result_output = ['New pass confirmation mismatch'];
	}
	else {
		$current_pass_arg = ' --current-pass=' . escapeshellarg($current_pass);
		$new_pass_arg = ' --new-pass=' . escapeshellarg($new_pass);
		$new_pass_confirmation_arg = ' --new-pass-confirmation=' . escapeshellarg($new_pass_confirmation);
		$result_code = null;
		$result_output = null;
		exec(THALASSA_CMD . ' --change-pass' . $current_pass_arg . $new_pass_arg . $new_pass_confirmation_arg, $result_output, $result_code);
	}
	$errors = [
		'Current pass not entered' => 'You must enter the current pass',
		'Incorrect pass' => 'Incorrect current pass',
		'New pass not entered' => 'You must enter and confirm a new pass',
		'New pass confirmation mismatch' => 'New pass and new pass confirmation didn\'t match',
	];
	# errors
	$error_r = [];
	if (0 != $result_code) {
		foreach ($result_output as $error) {
			if (-1 != strpos(', ', $error)) {
				$multierror = explode(', ', $error);
				foreach ($multierror as $error) {
					array_push($error_r, $errors[$error]);
				}
			}
			else {
				array_push($error_r, $errors[$error]);
			}
		}
	}
	return [$result_code, $error_r];
}
function list_tags($exclude_tags_string='', $exclude_future=false, $include_meta=false, $comma_separated_tags=false) {
	$exclude_tags_arg = '';
	if ($exclude_tags_string) {
		$exclude_tags_arg = ' --exclude-tags=' . escapeshellarg($exclude_tags_string);
	}

	$comma_separated_tags_arg = '';
	if ($comma_separated_tags) {
		$comma_separated_tags_arg = ' --comma-separated-tags';
	}

	$exclude_future_arg = '';
	if ($exclude_future) {
		$exclude_future_arg = ' --exclude-future';
	}

	$include_meta_arg = '';
	if ($include_meta) {
		$include_meta_arg = ' --include-meta';
	}

	$result_code = null;
	$result_output = null;
	exec(THALASSA_CMD . ' --list-tags' . $exclude_tags_arg . $comma_separated_tags_arg . $exclude_future_arg . $include_meta_arg, $result_output, $result_code);

	$tags_list = [];
	foreach ($result_output as $line) {
		$line_r = explode("\t", $line);
		$tag = $line_r[0];
		$count = $line_r[1];
		array_push($tags_list, [$tag, $count]);
	}
	return $tags_list;
}
function generate_tag_suggestions() {
	$tags_with_counts = list_tags('hidden', true, true);
	$tags = [];
	foreach ($tags_with_counts as $line) {
		array_push($tags, $line[0]);
	}
	file_put_contents(THALASSA_INFO['config_files_directory_path'] . 'tag_suggestions.json', json_encode(declutter_tags($tags)));

	$tags_full_with_counts = list_tags('', false, true);
	$tags_full = [];
	foreach ($tags_full_with_counts as $line) {
		array_push($tags_full, $line[0]);
	}
	file_put_contents(THALASSA_INFO['config_files_directory_path'] . 'tag_suggestions_full.json', json_encode(declutter_tags($tags_full)));

	file_put_contents(THALASSA_INFO['config_files_directory_path'] . 'last_tag_suggestions_generation.txt', time());
}
function list_tokens() {
	$result_code = null;
	$result_output = null;
	exec(THALASSA_CMD . ' --list-tokens', $result_output, $result_code);
	$tokens = json_decode($result_output[0], true);
	return $tokens;
}
function check_token($token) {
	$token_arg = '--token=' . escapeshellarg($token);
	$result_code = null;
	$result_output = null;
	exec(THALASSA_CMD . ' --check-token ' . $token_arg, $result_output, $result_code);
	return [$result_code, $result_output];
}function write_new_token() {
	$result_code = null;
	$result_output = null;
	exec(THALASSA_CMD . ' --write-new-token', $result_output, $result_code);
	return [$result_code, $result_output];
}
function set_tokens_mode($tokens_string, $token_mode_string) {
	$tokens_arg = '--tokens=' . escapeshellarg($tokens_string);
	$token_mode_arg = ' --token-mode=' . escapeshellarg($token_mode_string);
	$result_code = null;
	$result_output = null;
	exec(THALASSA_CMD . ' --set-tokens-mode ' . $tokens_arg . $token_mode_arg, $result_output, $result_code);
	return [$result_code, $result_output];
}
function set_token_name($token_string, $name_string) {
	$token_arg = ' --token=' . escapeshellarg($token_string);
	$name_arg = ' --name=' . escapeshellarg($name_string);
	$result_code = null;
	$result_output = null;
	exec(THALASSA_CMD . ' --set-token-name' . $token_arg . $name_arg, $result_output, $result_code);
	return [$result_code, $result_output];
}
function remove_tokens($tokens_string) {
	$tokens_arg = '--tokens=' . escapeshellarg($tokens_string);
	$result_code = null;
	$result_output = null;
	exec(THALASSA_CMD . ' --remove-tokens ' . $tokens_arg, $result_output, $result_code);
	return [$result_code, $result_output];
}
function remove_tags($tags_string, $comma_separated_tags=false) {
	$tags_arg = '--tags=' . escapeshellarg(format_tags_string($tags_string));
	$comma_separated_tags_arg = '';
	if ($comma_separated_tags) {
		$comma_separated_tags_arg = ' --comma-separated-tags';
	}
	$result_code = null;
	$result_output = null;
	exec(THALASSA_CMD . ' --remove-tags ' . $tags_arg . $comma_separated_tags_arg, $result_output, $result_code);
	return [$result_code, $result_output];
}
function replace_tag($old_tag, $new_tag) {
	$old_tag_arg = ' --old-tag=' . escapeshellarg(format_tags_string($old_tag));
	$new_tag_arg = ' --new-tag=' . escapeshellarg(format_tags_string($new_tag));
	$result_code = null;
	$result_output = null;
	exec(THALASSA_CMD . ' --replace-tag' . $old_tag_arg . $new_tag_arg, $result_output, $result_code);
	return [$result_code, $result_output];
}
function accompany_tag($tag, $new_tag) {
	#TODO remove early return when accompany_tag is finished in thalassa.py
	return
	$tag_arg = ' --tag=' . escapeshellarg(format_tags_string($tag));
	$new_tag_arg = ' --new-tag=' . escapeshellarg(format_tags_string($new_tag));
	$result_code = null;
	$result_output = null;
	exec(THALASSA_CMD . ' --accompany-tag' . $tag_arg . $new_tag_arg, $result_output, $result_code);
	return [$result_code, $result_output];
}
function set_first_pass($new_pass, $new_pass_confirmation) {
	$new_pass_arg = ' --new-pass=' . escapeshellarg($new_pass);
	$new_pass_confirmation_arg = ' --new-pass-confirmation=' . escapeshellarg($new_pass_confirmation);
	$result_code = null;
	$result_output = null;
	exec(THALASSA_CMD . ' --set-first-pass' . $new_pass_arg . $new_pass_confirmation_arg, $result_output, $result_code);
	return [$result_code, $result_output];
}
function initialize_database() {
	$result_code = null;
	$result_output = null;
	exec(THALASSA_CMD . ' --initialize-database', $result_output, $result_code);
	return [$result_code, $result_output];
}
function rebuild_all() {
	$result_code = null;
	$result_output = null;
	exec(THALASSA_CMD . ' --rebuild-all', $result_output, $result_code);
	return [$result_code, $result_output];
}
function process_temp_files(
	$temp_file_paths,
	$datetime_string,
	$tags_string,
	$comma_separated_tags,
	$tag_filenames,
	$filenames,
	$title='',
	$description=''
) {
	# consolidate local temp file paths for thalassa to process
	$temp_file_paths_string = implode(THALASSA_FILEPATHS_SEPARATOR, $temp_file_paths);
	$temp_file_paths_arg = ' --file-paths=' . escapeshellarg($temp_file_paths_string);

	$datetime_arg = ' --datetime=' . escapeshellarg($datetime_string);

	$additional_tags = '';
	if ('' != $title) {
		$additional_tags .= '#title:' . str_replace('#', '&#35;', $title);
	}
	if ('' != $description) {
		$additional_tags .= '#description:' . str_replace('#', '&#35;', $description);
	}

	$comma_separated_tags_arg = '';
	if ($comma_separated_tags) {
		$comma_separated_tags_arg = ' --comma-separated-tags';
		$additional_tags = str_replace('#', ',', $additional_tags);
	}

	$filenames_arg = '';
	if ($tag_filenames) {
		$filenames_string = implode(THALASSA_FILEPATHS_SEPARATOR, $filenames);
		$filenames_arg = ' --filenames=' . escapeshellarg($filenames_string);
	}
	$tags_arg = ' --tags=' . escapeshellarg(format_tags_string($tags_string . $additional_tags));

	$result_code = null;
	$result_output = null;
	exec(THALASSA_CMD . ' --store-files' . $temp_file_paths_arg . $tags_arg . $comma_separated_tags_arg . $filenames_arg . $datetime_arg, $result_output, $result_code);
	return([$result_code, $result_output]);
}
function search_files($tags_string, $page=null, $exclude_future=false) {
	if (array_key_exists('negate-tags', $_COOKIE) && $_COOKIE['negate-tags']) {
		$negate_tags = format_tags_string($_COOKIE['negate-tags']);
		$negate_tags = str_replace('#', '#-', $negate_tags);
		$tags_string .= $negate_tags;
	}

	# limit #perpage: for non-staff
	$tags_array = explode('#', format_tags_string($tags_string));
	if (0 < count($tags_array)) {
		if (!AUTHORIZED || !in_array(TOKEN_MODE, ['helper', 'admin'])) {
			foreach ($tags_array as $k => $tag) {
				if ('perpage:' == substr($tag, 0, 8)) {
					$perpage = (int)substr($tag, 8);
					if ($perpage > MAX_PERPAGE_NONSTAFF) {
						$tags_array[$k] = 'perpage:' . MAX_PERPAGE_NONSTAFF;
					}
					break;
				}
			}
		}
	}
	$tags_string = implode('#', $tags_array);

	$tags_arg = ' --tags=' . escapeshellarg($tags_string);

	$page_arg = '';
	if (null !== $page) {
		$page_arg = ' --page=' . escapeshellarg($page);
	}

	$exclude_future_arg = '';
	if ($exclude_future) {
		$exclude_future_arg = ' --exclude-future';
	}

	$result_code = null;
	$result_output = null;
	exec(THALASSA_CMD . ' --search-files' . $tags_arg . $page_arg . $exclude_future_arg, $result_output, $result_code);
	return [$result_code, $result_output];
}
function remove_files($file_ids_string) {
	$file_ids_arg = ' --file-ids=' . escapeshellarg($file_ids_string);
	$result_code = null;
	$result_output = null;
	exec(THALASSA_CMD . ' --remove-files' . $file_ids_arg, $result_output, $result_code);
	return [$result_code, $result_output];
}
function rebuild_files($file_ids_string) {
	$file_ids_arg = ' --file-ids=' . escapeshellarg($file_ids_string);
	$result_code = null;
	$result_output = null;
	exec(THALASSA_CMD . ' --rebuild' . $file_ids_arg, $result_output, $result_code);
	return [$result_code, $result_output];
}
function set_publish_time($file_ids_string, $datetime_string) {
	$file_ids_arg = ' --file-ids=' . escapeshellarg($file_ids_string);
	$datetime_arg = ' --datetime=' . escapeshellarg($datetime_string);
	$result_code = null;
	$result_output = null;
	exec(THALASSA_CMD . ' --set-publish-time' . $file_ids_arg . $datetime_arg, $result_output, $result_code);
	return [$result_code, $result_output];
}
function edit_files_tags($file_ids_string, $mode, $tags_string='', $comma_separated_tags=False) {
	$file_ids_arg = ' --file-ids=' . escapeshellarg($file_ids_string);
	$tags_arg = ' --tags=' . escapeshellarg(format_tags_string($tags_string));
	if (in_array($mode, ['replace', 'remove', 'add'])) {
		$mode = ' --' . $mode;
	}
	else {
		$mode = '';
	}

	$comma_separated_tags_arg = '';
	if ($comma_separated_tags) {
		$comma_separated_tags_arg = ' --comma-separated-tags';
	}

	$result_code = null;
	$result_output = null;
	$result = exec(THALASSA_CMD . ' --edit-files-tags' . $mode . $file_ids_arg . $tags_arg . $comma_separated_tags_arg, $result_output, $result_code);
	return [$result_code, $result_output];
}
function generate_set($file_ids_string, $sync=false) {
	$file_ids_arg = ' --file-ids=' . escapeshellarg($file_ids_string);
	$sync_arg = '';
	if ($sync) {
		$sync_arg = ' --sync';
	}
	$result_code = null;
	$result_output = null;
	exec(THALASSA_CMD . ' --generate-set' . $file_ids_arg . $sync_arg, $result_output, $result_code);
	return [$result_code, $result_output];
}
 ?>